%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: calc_uncertainty
%
% Scope:    calculate uncertainty for each isotopologue, plateau interval and gas type
%           associated with the applied corrections
%
% Input:    bindata (structure),
%           uncert (structure),
%           binref (structure),
%           list of gas names (cell),
%           list of gas type "flags" (i.e., MPV positions) (double),
%           uncertainty on slope N2O concentration correction (double),
%           uncertainty on slope CH4 spectral interference correction (double),
%           uncertainty on slope CO2 spectral interference correction (double),
%           anchor dataset name (string),
%           uncertainty due to other/poorly understood effects (double),
%           cal gas 1 name (string),
%           uncertainty true values cal gas 1 (double),
%           cal gas 2 name (string),
%           uncertainty true values cal gas 2 (double),
%
% Output:   updated "uncert" structure containing uncertainty values for each
%           isotopologue, plateau interval and gas type
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [uncert] = calc_uncertainty(bindata,uncert,binref,gasnames,gastypeflag,...
    uncert_slope_n2o,uncert_slope_ch4,uncert_slope_co2,anchor_gas,uncert_other,...
    true_gas1, true_values1, true_gas2, true_values2)

tic;fprintf('Running function calc_uncertainty...\n');
 
target_idx=4:6;     %d15Na, d15Nb, d18O index

label={'alpha','beta','o18'};
for z=1:3
    uncert.(label{1,z})=zeros(height(bindata.all),6);
end

uncert.check=1;             %flag for output file generation

%% print output on screen

if uncert.log(1,1) == 1         
    fprintf('Uncertainty on N2O concentration correction slope (alpha, beta, 18O)\n');
    fprintf('%1.3f, %1.3f, %1.3f\n',uncert_slope_n2o(1,1), uncert_slope_n2o(1,2), uncert_slope_n2o(1,3));
end

if uncert.log(1,2) == 1         
    fprintf('Uncertainty on CH4 concentration correction slope (alpha, beta, 18O)\n');
    fprintf('%1.3f, %1.3f, %1.3f\n',uncert_slope_ch4(1,1), uncert_slope_ch4(1,2), uncert_slope_ch4(1,3));
end

if uncert.log(1,3) == 1         
    fprintf('Uncertainty on CO2 concentration correction slope (alpha, beta, 18O)\n');
    fprintf('%1.3f, %1.3f, %1.3f\n',uncert_slope_co2(1,1), uncert_slope_co2(1,2), uncert_slope_co2(1,3));
end

labelp={'Uncertainty due to calibration correction (alpha)\n','Uncertainty due to calibration correction (beta)\n','Uncertainty due to calibration correction (18O)\n'};
for z=1:3        
    fprintf(labelp{1,z});
    if uncert.log(z,4)==2 
      fprintf('Type: 2-point calibration\n');    
      fprintf('Cal gas 1 = %s\n', true_gas1);
      fprintf('Uncertainty = %1.3f permil\n',true_values1(1,z));
      fprintf('Cal gas 2 = %s\n', true_gas2);
      fprintf('Uncertainty = %1.3f permil\n',true_values2(1,z));
   end           
   if uncert.log(z,4)==1 
      fprintf('Type: 1-point calibration\n');    
      fprintf('Cal gas 1 = %s\n', true_gas1);
      fprintf('Uncertainty = %1.3f permil\n',true_values1(1,z));
   end          
   if uncert.log(z,4)==0 
      fprintf('Type: No calibration');
   end        

end

fprintf('Uncertainty due to poorly understood effects (alpha, beta, 18O)\n');
fprintf('%1.3f, %1.3f, %1.3f\n',uncert_other(1,1), uncert_other(1,2), uncert_other(1,3));
    
%% calculate uncertainty for each isotopologue
    
for z=1:width(target_idx)
   
    if uncert.log(z,4) == 2     %2-point calibration
        
        %uncertainty due to N2O concentration correction 
        if uncert.log(z,1) == 1         
            bindata_idx=2;    %N2O index
            ind_anchor=find(ismember(gasnames,anchor_gas));
            for i=1:width(gastypeflag)
                    uncert.(label{1,z})(:,1)=bindata.calcorr.slope(z,1).*((1./bindata.all(:,bindata_idx))-(1./binref.(gasnames{1,ind_anchor})(1,bindata_idx)))*uncert_slope_n2o(1,z);                 
            end
        end
        
        %uncertainty due to CH4 spectral interference correction
        if uncert.log(z,2) == 1   
            bindata_idx=2;     %N2O index
            bindata_idx2=12;   %CH4 index
            ind_anchor=find(ismember(gasnames,anchor_gas));
            for i=1:width(gastypeflag)
                    uncert.(label{1,z})(:,2)=bindata.calcorr.slope(z,1).*((bindata.all(:,bindata_idx2)./bindata.all(:,bindata_idx))-(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx)))*uncert_slope_ch4(1,z);

            end
        end
        
        %uncertainty due to CO2 spectral interference correction
        if uncert.log(z,3) == 1   
            bindata_idx=2;     %N2O index
            bindata_idx2=14;   %CO2 index
            ind_anchor=find(ismember(gasnames,anchor_gas));
            %calculate uncertainty
            for i=1:width(gastypeflag)
                    uncert.(label{1,z})(:,3)=bindata.calcorr.slope(z,1).*((bindata.all(:,bindata_idx2)./bindata.all(:,bindata_idx))-(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx)))*uncert_slope_co2(1,z);
            end
        end
        
        %calibration correction (2-point)
        ind_gas1=find(ismember(gasnames,true_gas1));
        ind_gas2=find(ismember(gasnames,true_gas2));                
        for i=1:width(gastypeflag)
            factor1 = (bindata.calcorr.all(:,target_idx(1,z)) - bindata.calcorr_ref.(gasnames{1,ind_gas2})(1,target_idx(1,z))) ./ (bindata.calcorr_ref.(gasnames{1,ind_gas2})(1,target_idx(1,z)) - bindata.calcorr_ref.(gasnames{1,ind_gas1})(1,target_idx(1,z)));
            factor2 = (bindata.calcorr.all(:,target_idx(1,z)) - bindata.calcorr_ref.(gasnames{1,ind_gas1})(1,target_idx(1,z))) ./ (bindata.calcorr_ref.(gasnames{1,ind_gas2})(1,target_idx(1,z)) - bindata.calcorr_ref.(gasnames{1,ind_gas1})(1,target_idx(1,z)));                    
            uncert.(label{1,z})(:,4)=sqrt((factor1.*true_values1(1,z)).^2 + (factor2.*true_values2(1,z)).^2);
        end
        
    end
end

%% calculate uncertainty for each isotopologue (1-point calibration)

label={'alpha','beta','o18'};
    
for z=1:width(target_idx)
   
    if uncert.log(z,4) == 1  || uncert.log(z,4) == 0    %1-point calibration or No calibration
        
        %uncertainty due to N2O concentration correction
        if uncert.log(z,1) == 1         
            bindata_idx=2;    %N2O index
            ind_anchor=find(ismember(gasnames,anchor_gas));
            for i=1:width(gastypeflag)
                    uncert.(label{1,z})(:,1)=((1./bindata.all(:,bindata_idx))-(1./binref.(gasnames{1,ind_anchor})(1,bindata_idx)))*uncert_slope_n2o(1,z);
            end
        end
        
        %uncertainty due to CH4 spectral interference correction
        if uncert.log(z,2) == 1   
            bindata_idx=2;     %N2O index
            bindata_idx2=12;   %CH4 index
            ind_anchor=find(ismember(gasnames,anchor_gas));
            for i=1:width(gastypeflag)
                    uncert.(label{1,z})(:,2)=((bindata.all(:,bindata_idx2)./bindata.all(:,bindata_idx))-(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx)))*uncert_slope_ch4(1,z);

            end
        end
        
        %uncertainty due to CO2 spectral interference correction
        if uncert.log(z,3) == 1   
            bindata_idx=2;     %N2O index
            bindata_idx2=14;   %CO2 index
            ind_anchor=find(ismember(gasnames,anchor_gas));
            %calculate uncertainty
            for i=1:width(gastypeflag)
                    uncert.(label{1,z})(:,3)=((bindata.all(:,bindata_idx2)./bindata.all(:,bindata_idx))-(binref.(gasnames{1,ind_anchor})(1,bindata_idx2)./binref.(gasnames{1,ind_anchor})(1,bindata_idx)))*uncert_slope_co2(1,z);
            end
        end
    end

    if uncert.log(z,4) == 1  %1-point calibration
        %calibration correction (1-point)
        for i=1:width(gastypeflag)
            uncert.(label{:,z})(:,4)=true_values1(1,z);
        end
    end
        
end

%% uncertainty due to other effects (always included)
    
    for z=1:width(target_idx)
        for i=1:width(gastypeflag)
                uncert.(label{1,z})(:,5)=uncert_other(1,z);
        end
    end

%% calculate total uncertainty

    for z=1:width(target_idx)
        for k=1:height(bindata.all)
            uncert.(label{1,z})(k,6)=sqrt(sum(uncert.(label{1,z})(k,1:5).^2));
        end
    end
    
%% calc mean uncertainty over intervals

for z=1:width(target_idx)
    for k=1:height(bindata.boundaries_idx)-1
        for i=1:width(gastypeflag)
            if bindata.flag(k,1) == gastypeflag(1,i)
                uncert.mean.(label{1,z})(k,:)=mean(uncert.(label{1,z})(bindata.boundaries_idx(k,1)+1:bindata.boundaries_idx(k+1,1),:),'omitnan');
                %uncert.gastype(k,1)=gasnames(1,i);   
            end
        end
    end
end

%% re-define gas types (uncert)
  
for z=1:width(target_idx)
  for k=1:width(gasnames)
        uncert.(gasnames{1,k}).(label{1,z})=NaN(height(bindata.all),6);
        for i=1:height(bindata.all)
             if bindata.all(i,11)==gastypeflag(1,k)
                  uncert.(gasnames{1,k}).(label{1,z})(i,:)=uncert.(label{1,z})(i,:); 
             end
        end
  end
end
  
%% Remove empty rows (uncert)

for z=1:width(target_idx)
  for k=1:width(gasnames)
      uncert.(gasnames{1,k}).(label{1,z})=rmmissing(uncert.(gasnames{1,k}).(label{1,z}),1,'MinNumMissing',6);
  end
end

%% calc mean uncertainty over gas types

for z=1:width(target_idx)
    for k=1:6
        for i=1:width(gastypeflag)
                uncert.mean.(gasnames{1,i})(z,k)=mean(uncert.(gasnames{1,i}).(label{1,z})(:,k),'omitnan');
        end
    end
end

%% set uncertainty of N2O, CH4, CO2 corrections for anchor gas by definition equal to zero (small correction)

ind_anchor=find(ismember(gasnames,anchor_gas));
uncert.mean.(gasnames{1,ind_anchor})(:,1:3)=0;

for k=1:width(target_idx)
    uncert.mean.(gasnames{1,ind_anchor})(k,6)=sqrt(sum(uncert.mean.(gasnames{1,ind_anchor})(k,1:5).^2));
end

for z=1:width(target_idx)
    if uncert.log(z,4) == 2     %2-point calibration
        ind_gas1=find(ismember(gasnames,true_gas1));
        ind_gas2=find(ismember(gasnames,true_gas2));       
         uncert.mean.(gasnames{1,ind_gas1})(z,4)=true_values1(1,z);   
         uncert.mean.(gasnames{1,ind_gas2})(z,4)=true_values2(1,z);   
    end
end

for k=1:width(target_idx)
    uncert.mean.(gasnames{1,ind_gas1})(k,6)=sqrt(sum(uncert.mean.(gasnames{1,ind_gas1})(k,1:5).^2));
    uncert.mean.(gasnames{1,ind_gas2})(k,6)=sqrt(sum(uncert.mean.(gasnames{1,ind_gas2})(k,1:5).^2));
end

%% Take absolute value

for i=1:width(gastypeflag)
        uncert.mean.(gasnames{1,i})=abs(uncert.mean.(gasnames{1,i}));
end

for z=1:width(target_idx)
        uncert.mean.(label{1,z})=abs(uncert.mean.(label{1,z}));
end

%%
time_elapsed=toc; fprintf('calc_uncertainty completed (execution time: %1.2f s)\n',time_elapsed); 

end

